// =============================================================================
//                           COPYRIGHT NOTICE
// Copyright 2006 (c) Lattice Semiconductor Corporation
// ALL RIGHTS RESERVED
// This confidential and proprietary software may be used only as authorised by
// a licensing agreement from Lattice Semiconductor Corporation.
// The entire notice above must be reproduced on all authorized copies and
// copies may only be made to the extent permitted by a licensing agreement from
// Lattice Semiconductor Corporation.
//
// Lattice Semiconductor Corporation        TEL : 1-800-Lattice (USA and Canada)
// 5555 NE Moore Court                            408-826-6000 (other locations)
// Hillsboro, OR 97124                     web  : http://www.latticesemi.com/
// U.S.A                                   email: techsupport@latticesemi.com
// =============================================================================/
//                         FILE DETAILS
// Project          : LM32 DMA Component
// File             : slave_reg.v
// Title            : DMA Slave controller 
// Dependencies     : None
// Version          : 7.0
//                  : Initial Release
// Version          : 7.0SP2, 3.0
//   1. Read and Write channel of DMA controller are working in parallel,
//      due to that now as soon as FIFO is not empty write channel of the DMA
//      controller start writing data to the slave.
//   2. Burst Size supported by DMA controller is increased to support bigger
//      burst (from current value of 4 and 8 to 16 and 32). Now 4 different type
//      of burst sizes are supported by the DMA controller 4, 8, 16 and 32. 
//      For this Burst Size field of the control register is increased to 2 bits.
//   3. Glitch is removed on the S_ACK_O signal. 
// Version          : 3.1
//                  : Make DMA Engine compliant to Rule 3.100 of Wishbone Spec
//                  : which defines alignement of bytes in sub-word transfers.
// =============================================================================

`ifndef SLAVE_REG_FILE
 `define SLAVE_REG_FILE
 `include "system_conf.v"
module SLAVE_REG 
  #(parameter LENGTH_WIDTH = 16,
    parameter FIFO_IMPLEMENTATION = "EBR")
    (
     //slave port
     S_ADR_I,    //32bits
     S_DAT_I,    //32bits
     S_WE_I,
     S_STB_I,
     S_CYC_I,
     S_CTI_I,
     S_DAT_O,    //32bits
     S_ACK_O,
     S_INT_O,
     //Master Address
//      MA_SEL_O,
//      MB_SEL_O,
     M_SEL_O,
     //internal signals
     reg_start,
     reg_status,
     reg_interrupt,
     reg_busy,
     data_length,
     reg_cntlg,
     reg_bt2,reg_bt1,reg_bt0,
     incr_unit,
     reg_s_con,
     reg_d_con,
     reg_00_data,
     reg_04_data,
     //system clock and reset
     CLK_I,
     RST_I
     );

   input [31:0]    S_ADR_I;
   input [31:0]    S_DAT_I;    //32bits
   input           S_WE_I;
   input           S_STB_I;
   input           S_CYC_I;
   input [2:0]     S_CTI_I;
   output [31:0]   S_DAT_O;    //32bits
   output          S_ACK_O;
   output          S_INT_O;    //interrupt signal
   //Master Address
   output [3:0] M_SEL_O;
//    output [3:0]    MA_SEL_O;
//    output [3:0]    MB_SEL_O;
   //internal signals
   output          reg_start;
   input           reg_status;
   input           reg_interrupt;
   input           reg_busy;
   output [LENGTH_WIDTH-1:0] data_length;
   input                     reg_cntlg;
   output                    reg_bt2,reg_bt1,reg_bt0;
   output [2:0]              incr_unit;
   output                    reg_s_con;
   output                    reg_d_con;
   output [31:0]             reg_00_data;
   output [31:0]             reg_04_data;

   //system clock and reset
   input                     CLK_I;
   input                     RST_I;

   parameter                 UDLY = 1;

   reg [31:0]                reg_00_data;
   reg [31:0]                reg_04_data;
   reg [LENGTH_WIDTH-1:0]    reg_08_data;
   reg [6:0]                 reg_0c_data;

   reg [3:0]                 M_SEL_O;
//    wire [3:0]                MA_SEL_O    = M_SEL_O;
//    wire [3:0]                MB_SEL_O    = M_SEL_O;
   wire [LENGTH_WIDTH-1:0]   data_length    = reg_08_data;

   wire                      reg_bt2, reg_bt1, reg_bt0, reg_incw, reg_inchw, reg_d_con, reg_s_con;
   assign                    {reg_bt2,reg_bt1,reg_bt0,reg_incw,reg_inchw,reg_d_con,reg_s_con} = reg_0c_data;
   wire [2:0]                incr_unit = reg_incw ? 4 : reg_inchw ? 2 : 1;

   wire [8:0]                burst_incr_unit = reg_bt2 ? (reg_bt1 ? (reg_bt0 ? incr_unit<<5 : incr_unit<<4) : (reg_bt0 ? incr_unit<<3 : incr_unit<<2)) : incr_unit;
   reg                       reg_ie;
   wire [2:0]                read_10_data    = {reg_status,reg_ie,reg_busy};

   wire                      reg_wr_rd    = S_CYC_I && S_STB_I;

   wire                      master_idle = !reg_busy;
   reg                       s_ack_o_pre;
   wire                      S_ACK_O    = s_ack_o_pre  && S_CYC_I && S_STB_I;
   
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       s_ack_o_pre         <= #UDLY 1'b0;
     else if(((master_idle && reg_wr_rd) || (!master_idle && reg_wr_rd && !S_WE_I)) && (!s_ack_o_pre)) 
       s_ack_o_pre         <= #UDLY 1'b1;
     else	     
       s_ack_o_pre         <= #UDLY 1'b0;


   //register write and read
   wire                      reg_wr          = reg_wr_rd && S_WE_I && master_idle && S_ACK_O;
   wire                      reg_rd          = reg_wr_rd && !S_WE_I && S_ACK_O;

   wire                      dw00_cs         = (!(|S_ADR_I[5:2]));
   wire                      dw04_cs         = (S_ADR_I[5:2] == 4'h1);
   wire                      dw08_cs         = (S_ADR_I[5:2] == 4'h2);
   wire                      dw0c_cs         = (S_ADR_I[5:2] == 4'h3);
   wire                      dw10_cs         = (S_ADR_I[5:2] == 4'h4);

   //S_DAT_O
   wire [31:0]               S_DAT_O = dw00_cs ? reg_00_data :
                             dw04_cs ? reg_04_data :
                             dw08_cs ? reg_08_data :
                             dw0c_cs ? {24'h0,1'h0,reg_0c_data} :
                             dw10_cs ? {24'h0,5'h0,read_10_data} : 32'h0;

   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       M_SEL_O             <= #UDLY 4'h0;
     else if(data_length < incr_unit)
       case(data_length[2:0])
         1:    M_SEL_O     <= #UDLY 4'h8;
         2:    M_SEL_O     <= #UDLY 4'hc;
         3:    M_SEL_O     <= #UDLY 4'he;
         default:M_SEL_O   <= #UDLY 4'hf;
       endcase
     else
       case(incr_unit)
         1:    M_SEL_O     <= #UDLY 4'h8;
         2:    M_SEL_O     <= #UDLY 4'hc;
         4:    M_SEL_O     <= #UDLY 4'hf;
         default:M_SEL_O   <= #UDLY 4'hf;
       endcase
   //interrupt
   reg                       S_INT_O;
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       S_INT_O             <= #UDLY 1'b0;
     else if(reg_interrupt && reg_ie)
       S_INT_O             <= #UDLY 1'b1;
     else if(dw10_cs && reg_rd)
       S_INT_O             <= #UDLY 1'b0;

   //reg_00
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       reg_00_data         <= #UDLY 32'h0;
     else if(dw00_cs && reg_wr)
       reg_00_data         <= #UDLY S_DAT_I;

   //reg_04
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       reg_04_data         <= #UDLY 32'h0;
     else if(dw04_cs && reg_wr)
       reg_04_data         <= #UDLY S_DAT_I;

   //reg_08
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       reg_08_data         <= #UDLY 32'h0;
     else if(reg_cntlg)
       reg_08_data         <= #UDLY (reg_08_data < burst_incr_unit) ? 'h0 : (reg_08_data - burst_incr_unit);
     else if(dw08_cs && reg_wr)
       reg_08_data         <= #UDLY S_DAT_I;

   //reg_0c
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       reg_0c_data         <= #UDLY 7'h0;
     else if(dw0c_cs && reg_wr)
       reg_0c_data         <= #UDLY S_DAT_I[6:0];

   //reg_10
   reg                       reg_start;
   always @(posedge CLK_I or posedge RST_I)
     if(RST_I)
       begin
          reg_ie           <= #UDLY 1'b0;
          reg_start        <= #UDLY 1'b0;
       end 
     else if(dw10_cs && reg_wr) 
       begin
          reg_ie           <= #UDLY S_DAT_I[1];
          reg_start        <= #UDLY S_DAT_I[3];
       end 
     else 
       begin
          reg_start        <= #UDLY 1'b0;
       end
endmodule // SLAVE_REG
`endif // SLAVE_REG_FILE
