/****************************************************************************
 *
 *
 ****************************************************************************/
 
module wb_sdram (
	// Clocks and resets
	input						wb_clk_i,			// WISHBONE clock
	input						wb_rst_i,			// WISHBONE reset

	// WISHBONE bus
	input			[31:0]	wb_adr_i,			// WISHBONE address
	input			[31:0]	wb_dat_i,			// WISHBONE data in
	output reg	[31:0]	wb_dat_o,			// WISHBONE data out
	input			[3:0]		wb_sel_i,			// WISHBONE byte select
	input						wb_we_i,				// WISHBONE write enable (R/#W)
	input						wb_cyc_i,			// WISHBONE cycle
	input						wb_stb_i,			// WISHBONE strobe
	output					wb_ack_o,			// WISHBONE cycle acknowledge (data available, DTACK)
	output					wb_err_o,			// WISHBONE bus error
	output					wb_rty_o,			// WISHBONE retry-later
	
	// SDRAM
	output reg				sdram_cke,			// SDRAM clock enable
	output					sdram_cs_n,			// SDRAM chip select (active low)
	output					sdram_ras_n,		// SDRAM row address strobe (active low)
	output					sdram_cas_n,		// SDRAM column address strobe (active low)
	output					sdram_we_n,			// SDRAM write enable (active low)
	output		[11:0]	sdram_a,				// SDRAM address
	output reg	[1:0]		sdram_ba,			// SDRAM bank address
	output reg	[3:0]		sdram_dqm,			// SDRAM data mask (OE#; 0=active, 1=disabled)
	inout			[31:0]	sdram_dq,			// SDRAM data bus

	// Debugging
	output /*reg*/	[2:0]		debug					// debug bits
);


/****
 * SDRAM data output buffer
 ****/
// OE=1 for output mode, 0 for input
reg sdram_dq_oe;
// SDRAM output register
reg [31:0] sdram_dq_r;
assign sdram_dq = sdram_dq_oe ? sdram_dq_r : 32'hZZZZ;



/****
 * State timer
 * This is used to ensure that the state machine abides by RAM timing
 * restrictions.
 ****/
reg [31:0] timer;


/****
 * MODE logic
 ****/
reg  [5:0]  sdram_mode;
reg  [11:0]	sdram_addr;
assign sdram_cs_n		= sdram_mode[3];
assign sdram_ras_n	= sdram_mode[2];
assign sdram_cas_n	= sdram_mode[1];
assign sdram_we_n		= sdram_mode[0];
assign sdram_a = {sdram_addr[11], (sdram_mode[5] ? sdram_mode[4] : sdram_addr[10]), sdram_addr[9:0]};

// SDRAM chip instructions
// The bit order is as specified in the ISSI datasheet: A10 Override, A10, CS#, RAS#, CAS#, WE#.
// If A10 Override is set, then A10 will be overridden to the value specified in the M_ constant.
localparam M_BankActivate		= 6'b0X0011;
localparam M_PrechargeBank		= 6'b100010;
localparam M_PrechargeAll		= 6'b110010;
localparam M_Write				= 6'b100100;
localparam M_WritePrecharge	= 6'b110100;
localparam M_Read					= 6'b100101;
localparam M_ReadPrecharge		= 6'b110101;
localparam M_LoadModeRegister	= 6'b0X0000;
localparam M_Nop					= 6'b0X0111;
localparam M_BurstStop			= 6'b0X0110;
localparam M_Inhibit				= 6'b0X1XXX;		// maybe X1111?
localparam M_AutoRefresh		= 6'b0X0001;


/****
 * Refresh Timer
 ****/
parameter REFRESH_INTERVAL = 32'd390 - 32'd1;
reg [31:0] refresh_timer;
reg refresh_req, refresh_ack, refresh_timer_en;
always @(posedge wb_clk_i) begin
	if (wb_rst_i | !refresh_timer_en) begin
		// Reset; clear timer, unset REFRESH REQUEST
		refresh_req <= 1'b0;
		refresh_timer <= REFRESH_INTERVAL;
	end else if (refresh_ack) begin
		// Refresh Ack, clear Refresh Request.
		refresh_req <= 1'b0;
	end else if (refresh_timer == 0) begin
		// Refresh timer timed out, make a Refresh Request and reload the timer
		refresh_req <= 1'b1;
		refresh_timer <= REFRESH_INTERVAL;
	end else begin
		// Otherwise just decrement the timer
		refresh_timer <= refresh_timer - 32'd1;
	end
end

assign debug = { 1'b0, refresh_req, refresh_ack };

/****
 * Finite State Machine
 ****/
localparam	ST_INIT1							= 32'd0;
localparam	ST_INIT2							= 32'd1;
localparam	ST_NOP1							= 32'd2;
localparam	ST_PrechargeAll				= 32'd3;
localparam	ST_PrechargeAll_Wait			= 32'd4;
localparam	ST_AutoRefresh1				= 32'd5;
localparam	ST_AutoRefresh1_Wait			= 32'd6;
localparam	ST_AutoRefresh2				= 32'd7;
localparam	ST_AutoRefresh2_Wait			= 32'd8;
localparam	ST_LoadModeRegister			= 32'd9;
localparam	ST_LoadModeRegister_Wait	= 32'd10;
localparam	ST_Spin							= 32'd11;		// <<== main 'spin' / 'idle' state
localparam	ST_Refresh						= 32'd12;
localparam	ST_Refresh_Wait				= 32'd13;
 
reg [31:0] state;
always @(posedge wb_clk_i) begin
	if (wb_rst_i) begin
		// Initialise state machine and timer
		state <= ST_INIT1;
//		debug <= 3'd0;
		timer <= 32'd0;
		
		// Clear REFRESH ACK flag and disable refresh timer
		refresh_ack <= 1'b0;
		refresh_timer_en <= 1'b0;

		// Initialisation state for SDRAM
		sdram_cke	<= 1'b0;
		sdram_mode	<= M_Inhibit;
		sdram_addr	<= 12'h000;
		sdram_ba		<= 2'b00;
		sdram_dqm	<= 4'b0000;
		sdram_dq_oe	<= 1'b0;			// data output disabled
		sdram_dq_r	<= 32'd0;
	end else begin
		// timer logic
		if (timer > 32'd0) begin
			timer <= timer - 32'd1;
		end
	
		// state machine logic
		case (state)
			ST_INIT1: begin
					// INIT1: Set up for initial power-up wait
					state <= ST_INIT2;
					timer <= 32'd50_000;		// TODO: dependent on core clock rate. Needs to be >= 100us
					
					// SDRAM state
					sdram_cke	<= 1'b0;			// clock disabled
					sdram_mode	<= M_Inhibit;
					sdram_addr	<= 12'h000;
					sdram_ba		<= 2'b00;
					sdram_dqm	<= 4'b1111;
					sdram_dq_oe	<= 1'b0;			// data output disabled
					sdram_dq_r	<= 32'd0;
				end
				
			ST_INIT2: begin
					// INIT2: Power-up wait. Keep CKE low until ~50 cycles before
					// the end of the power-up wait, then bring CKE high.
					if (timer == 32'd0) begin
						// Timer hit zero. Send a NOP.
						state <= ST_NOP1;
					end else if (timer < 32'd50) begin
						// Timer value is more than zero but less than 50; CKE is on, but
						// keep waiting for the timer to actually expire.
						sdram_cke	<= 1'b1;
						state			<= ST_INIT2;
//						debug <= 3'd1;
					end
					sdram_mode <= M_Inhibit;
				end
				
			ST_NOP1: begin
					// Apply one or more NOP commands to the SDRAM
					sdram_mode <= M_Nop;
					state <= ST_PrechargeAll;
				end
				
			ST_PrechargeAll: begin
					// Precharge All, then wait T_rp (20ns)
					sdram_mode <= M_PrechargeAll;
					timer <= 32'd0;	// wait 1tcy (40ns)   ---> TIMER HERE
					state <= ST_PrechargeAll_Wait;
				end

			ST_PrechargeAll_Wait: begin
					// Wait for T_rp after Precharge All
					sdram_mode <= M_Nop;
					if (timer == 32'd0) begin
						// Timer hit zero. Continue
						state <= ST_AutoRefresh1;
					end
				end
				
			ST_AutoRefresh1: begin
					// Auto Refresh 1 of 2, wait T_rfc (70ns) after each
					sdram_mode <= M_AutoRefresh;
					timer <= 32'd1;	// wait 2tcy (80ns)  ---> TIMER HERE
					state <= ST_AutoRefresh1_Wait;
				end
				
			ST_AutoRefresh1_Wait: begin
					// Wait for T_rfc
					sdram_mode <= M_Nop;
					if (timer == 32'd0) begin
						// Timer hit zero. Continue
						state <= ST_AutoRefresh2;
					end
				end
				
			ST_AutoRefresh2: begin
					// Auto Refresh 2 of 2, wait T_rfc (70ns) after each
					sdram_mode <= M_AutoRefresh;
					timer <= 32'd1;	// wait 2tcy (80ns)  ---> TIMER HERE
					state <= ST_AutoRefresh2_Wait;
				end
				
			ST_AutoRefresh2_Wait: begin
					// Wait for T_rfc
					sdram_mode <= M_Nop;
					if (timer == 32'd0) begin
						// Timer hit zero. Continue
						state <= ST_LoadModeRegister;
					end
				end

			ST_LoadModeRegister: begin
					// Load Mode Register
					/**
					 * Mode register:
					 *   - BS0,1  = 00  [RFU]
					 *   - A11,10 = 00  [RFU]
					 *   - A9     = 0   [WBL -- write burst length same as read burst length]
					 *   - A8,7   = 00  [Test Mode off]
					 *   - A6..4  = 010 [CAS Latency = 2 clocks]
					 *   - A3     = 0   [Burst type = sequential]
					 *   - A2..0  = 000 [Burst length = 1 word]
					 */
					sdram_ba <= 2'b00;
					sdram_addr <= 12'b00_0_00_010_000;
					sdram_mode <= M_LoadModeRegister;

					// Wait T_mrd (2 clock cycles)
					timer <= 32'd1;	// (2cy)-1  ---> TIMER HERE
					state <= ST_LoadModeRegister_Wait;
				end

			ST_LoadModeRegister_Wait: begin
					// Wait for LMR to complete
					sdram_mode <= M_Nop;
					sdram_ba <= 2'd0;
					sdram_addr <= 12'd0;
					if (timer == 32'd0) begin
						// Timer hit zero. Continue
						state <= ST_Spin;
					end
				end

			ST_Spin: begin
					// Enable refresh timer
					refresh_timer_en <= 1'b1;

					sdram_mode <= M_Inhibit;

					if (refresh_req) begin
						// Refresh request received. Ack it and do a refresh.
						refresh_ack <= 1'b1;
						state <= ST_Refresh;
					end else begin
						state <= ST_Spin;
					end
				end
				
			ST_Refresh: begin
					// Refresh timer timed out; do a refresh run
					// Start by clearing the ACK flag (which was set by the Spin state)
					refresh_ack <= 1'b0;
					// Tell the SDRAM to do a Refresh
					sdram_mode <= M_AutoRefresh;
					// Wait for T_rfc
					timer <= 32'd1;	// wait Trfc (70ns ideally, we give 80ns)  ---> TIMER HERE
					state <= ST_Refresh_Wait;
				end
				
			ST_Refresh_Wait: begin
					// Wait for T_rfc
					sdram_mode <= M_Nop;
					if (timer == 32'd0) begin
						// Timer hit zero. Go back to spin state.
						state <= ST_Spin;
					end
				end
		endcase
	end
end

endmodule
