/*
 * tumble: build a PDF file from image files
 *
 * Copyright 2004 Daniel Gloeckner
 *
 * Derived from tumble_jpeg.c written 2003 by Eric Smith <eric at brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
 */


#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <math.h>
#include <strings.h>  /* strcasecmp() is a BSDism */


#include "semantics.h"
#include "tumble.h"
#include "bitblt.h"
#include "pdf.h"
#include "tumble_input.h"


static bool match_jp2_suffix (char *suffix)
{
  return (strcasecmp (suffix, ".jp2") == 0);
}

static bool close_jp2_input_file (void)
{
  return (1);
}

static struct {
  FILE *f;
  uint32_t width,height;
  struct {
    double VR,HR;
  } res[2];
} jp2info;

struct box {
  char TBox[4];
  bool (*func)(uint64_t size, void *cparam);
  void *cparam;
};

#define BE32(p) (((p)[0]<<24)+((p)[1]<<16)+((p)[2]<<8)+(p)[3])
#define BE16(p) (((p)[2]<<8)+(p)[3])

static bool skipbox(uint64_t size, void *cparam)
{
  if(size==~0)
    fseek(jp2info.f,0,SEEK_END);
  else {
    if(cparam)
      size-=*(int *)cparam;
    fseek(jp2info.f,size,SEEK_CUR); /*FIXME: size is 64bit*/
  }
  return 1;
}

static bool read_res(uint64_t size, void *cparam)
{
  int read=0;
  bool ret=1;
  if(size>=10) {
    int i;
    unsigned char buf[10];

    ret=0;
    i=(int)cparam;
    read=fread(buf,1,10,jp2info.f);
    if(read==10) {
      uint16_t vrn,vrd,hrn,hrd;
      int8_t vre,hre;
      vrn=BE16(buf);
      vrd=BE16(buf+2);
      hrn=BE16(buf+4);
      hrd=BE16(buf+6);
      vre=((signed char *)buf)[8];
      hre=((signed char *)buf)[9];
      if(vrn && vrd && hrn && hrd) {
	jp2info.res[i].VR=vrn*pow(10.0,vre)/vrd;
	jp2info.res[i].HR=hrn*pow(10.0,hre)/hrd;
	ret=1;
      }
    }
  }
  skipbox(size,&read);
  return ret;
}

static bool read_ihdr(uint64_t size, void *cparam)
{
  int read=0;
  bool ret=1;
  if(size>=8) {
    unsigned char buf[8];
    read=fread(buf,1,8,jp2info.f);
    if(read==8) {
      jp2info.height=BE32(buf);
      jp2info.width=BE32(buf+4);
    } else
      ret=0;
  }
  skipbox(size,&read);
  return ret;
}

static bool superbox(uint64_t size, void *cparam)
{ 
  while(size>=8) {
    static unsigned char buf[12];
    int r;
    uint64_t s;
    struct box *l;

    r=fread(buf+4,1,8,jp2info.f);
    if(r<8)
      return (size==~0 && r==0);
 
    s=BE32(buf+4);
    if(s==1 && size>=16) {
      r=fread(buf,1,8,jp2info.f);
      if(r<8)
        return 0;
      s=((uint64_t)BE32(buf)<<32)+BE32(buf+4);
    }
    if(s && s<8)
      return 0;
    if(size!=~0) {
      if(!s)
        return 0;
      size-=s;
    } else if(!s)
      size=0;
    s=s?s-8:~0;

    for(l=(struct box *)cparam;l->func;l++)
      if(!memcmp(l->TBox,buf+8,4))
        break;
    if(l->func) {
      if(!l->func(s,l->cparam))
	return 0;
    }else
      if(!skipbox(s,NULL))
	return 0;
  }
  
  return size==0;
}

static const struct box res_boxes[]={
  {"resc",read_res,(void *)0},
  {"resd",read_res,(void *)1},
  {"",NULL,NULL}
};

static const struct box jp2h_boxes[]={
  {"ihdr",read_ihdr,NULL},
  {"res ",superbox,(void *)res_boxes},
  {"",NULL,NULL}
};

static const struct box root[]={
  {"jp2h",superbox,(void *)jp2h_boxes},
  {"",NULL,NULL}
};

static bool open_jp2_input_file (FILE *f, char *name)
{
  int s;
  const char sig[12]="\0\0\0\xCjP  \r\n\x87\n";
  char buf[12];

  memset(&jp2info,0,sizeof(jp2info));
  jp2info.f=f;
  
  s=fread(buf,1,12,f);
  rewind(f);
  return (s==12 && !memcmp(buf,sig,12));
}


static bool last_jp2_input_page (void)
{
  return 1;
}


static bool get_jp2_image_info (int image,
				 input_attributes_t input_attributes,
				 image_info_t *image_info)
{
  int i;

  if(!superbox(~0,(void *)root))
    return 0;
  rewind(jp2info.f);
  
#ifdef DEBUG_JPEG
  printf ("capture x density: %d pixel/m\n", jp2info.res[0].HR);
  printf ("capture y density: %d pixel/m\n", jp2info.res[0].VR);
  printf ("display x density: %d pixel/m\n", jp2info.res[1].HR);
  printf ("display y density: %d pixel/m\n", jp2info.res[1].VR);
  printf ("width: %d\n", jp2info.width);
  printf ("height: %d\n", jp2info.height);
#endif

  image_info->color = 1;
  image_info->width_samples = jp2info.width;
  image_info->height_samples = jp2info.height;

  image_info->width_points = (image_info->width_samples * POINTS_PER_INCH) / 300.0;
  image_info->height_points = (image_info->height_samples * POINTS_PER_INCH) / 300.0;

  for(i=2;i--;)
    if(jp2info.res[i].VR > 0.0 && jp2info.res[i].HR > 0.0) {
      image_info->width_points = (image_info->width_samples * POINTS_PER_INCH * 10000)/(jp2info.res[i].HR * 254);
      image_info->height_points = (image_info->height_samples * POINTS_PER_INCH * 10000)/(jp2info.res[i].VR * 254);
      break;
    }

  return 1;
}


static bool process_jp2_image (int image,  /* range 1 .. n */
				input_attributes_t input_attributes,
				image_info_t *image_info,
				pdf_page_handle page)
{
  pdf_write_jp2_image (page,
			0, 0,  /* x, y */
			image_info->width_points,
			image_info->height_points,
			image_info->width_samples,
			image_info->height_samples,
			jp2info.f);

  return (1);
}


input_handler_t jp2_handler =
  {
    match_jp2_suffix,
    open_jp2_input_file,
    close_jp2_input_file,
    last_jp2_input_page,
    get_jp2_image_info,
    process_jp2_image
  };


void init_jp2_handler (void)
{
  install_input_handler (& jp2_handler);
}
